<?php

namespace App\DataTables;

use App\Models\User;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class DoctorsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *

     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $dataTable = new EloquentDataTable($query);

        return $dataTable
            ->addColumn('specialty', function (User $doctor) {
                return $doctor->doctorProfile->specialty->name ?? '-';
            })
            ->addColumn('clinic', function (User $doctor) {
                return $doctor->clinic?->name ?? '-';
            })
            ->addColumn('branch', function (User $doctor) {
                return $doctor->branch?->name ?? '-';
            })
            ->addColumn('is_active', function (User $doctor) {
                $checked = $doctor->is_active ? 'checked' : '';

                return '
        <label class="switch">
            <input type="checkbox" class="toggle-status"
                   data-id="' . $doctor->id . '"
                   ' . $checked . '>
            <span class="slider round"></span>
        </label>
    ';
            })
            ->addColumn('action', function (User $doctor) {
                return view('admin.doctors.datatables.actions', compact('doctor'));
            })
            ->rawColumns(['action', 'clinic', 'branch', 'is_active', 'specialty']);
    }

    /**
     * Get the query source of dataTable.
     *

     */
    public function query(User $model): QueryBuilder
    {
        $user = Auth::user();

        $query = $model->newQuery()
            ->orderByDesc('id')
            ->where('role', 'doctor')
            ->with('clinic:id,name', 'branch:id,name')
            ->select([
                'id',
                'clinic_id',
                'branch_id',
                'name',
                'email',
                'phone',
                'is_active',
            ]);


        if ($user && $user->clinic_id) {
            $query->where('clinic_id', $user->clinic_id);

            if ($user->branch_id) {
                $query->where('branch_id', $user->branch_id);
            }
        }

        return $query;
    }

    /**
     * Optional method if you want to use the html builder.
     */


    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('doctors-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->parameters($this->getParameters());
    }

    protected function getParameters()
    {
        return [
            'dom' => 'Blfrtip',
            'lengthMenu' => [
                [10, 25, 50, 100, 500, 750, -1],
                ['10', '25 ', '50 ', '100 ', '500 ', '750', trans('dashboard/datatable.all_records')]
            ],
            'buttons' => [
                [
                    'extend' => 'csv',
                    'className' => 'btn btn-primary',
                    'text' => "<i class='fa fa-file'></i>" . " تصدير ك  Excel"
                ],
                [
                    'extend' => 'excel',
                    'className' => 'btn btn-success',
                    'text' => "<i class='fa fa-file'></i>" . "تصدير ك Excel "
                ],
                [
                    'extend' => 'print',
                    'className' => 'btn btn-info',
                    'text' => "<i class='fa fa-print'></i>" . "طباعة "
                ],
                /*[
                    'extend' => 'reload',
                    'className' => 'btn btn-dark',
                    'text' => "<i class='fa fa-sync-alt'></i>" . trans('dashboard/datatable.reload')
                ]*/
            ],

            'columnDefs' => [
                ['className' => 'text-center', 'targets' => '_all'],
            ],
        ];
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [


            Column::make('id')->title('#'),
            Column::make('name')->title('اسم الدكتور'),
            Column::make('phone')->title('الجوال'),

            Column::make('clinic')->title('العيادة'),
            Column::make('branch')->title('الفرع'),
            Column::make('specialty')->title('التخصص'),
            Column::make('is_active')->title('الحالة'),


            Column::computed('action')
                ->title('إجراءات')
                ->exportable(false)
                ->printable(false)
                ->width(80)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Doctors_' . date('YmdHis');
    }
}
