<?php

namespace App\DataTables;


use App\Models\Patients;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class PatientsDataTable extends DataTable
{
    /**
     * Build DataTable class.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        return (new EloquentDataTable($query))
            ->addColumn('full_name', function (Patients $patient) {
                return $patient->first_name . ' ' . $patient->last_name;
            })
            ->addColumn('actions', function (Patients $patient) {
                return view('admin.patients.datatables.actions', compact('patient'));
            })
            ->rawColumns(['full_name','actions']);
    }

    /**
     * Get query source.
     */
    public function query(Patients $model): QueryBuilder
    {
        $user = auth()->user();


        $query = $model->newQuery()
            ->orderByDesc('id')
            ->with('clinic:id,name', 'branch:id,name')
            ->select([
                'id',
                'clinic_id',
                'branch_id',
                'code',
                'first_name',
                'last_name',
                'phone',
                'gender',
                'birth_date',
                'created_at',
            ]);

        if ($user && $user->clinic_id) {
            $query->where('clinic_id', $user->clinic_id);
        }

        if ($user && $user->role !== 'owner' && $user->branch_id) {
            $query->where('branch_id', $user->branch_id);
        }

        return $query;
    }

    /**
     * HTML builder.
     */


    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('patients-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->parameters($this->getParameters());
    }


    protected function getParameters()
    {
        return [
            'dom' => 'Blfrtip',
            'lengthMenu' => [
                [10, 25, 50, 100, 500, 750, -1],
                ['10', '25 ', '50 ', '100 ', '500 ', '750', trans('dashboard/datatable.all_records')]
            ],
            'buttons' => [
                [
                    'extend' => 'csv',
                    'className' => 'btn btn-primary',
                    'text' => "<i class='fa fa-file'></i>" . " تصدير ك  Excel"
                ],
                [
                    'extend' => 'excel',
                    'className' => 'btn btn-success',
                    'text' => "<i class='fa fa-file'></i>" . "تصدير ك Excel "
                ],
                [
                    'extend' => 'print',
                    'className' => 'btn btn-info',
                    'text' => "<i class='fa fa-print'></i>" ."طباعة "
                ],
                /*[
                    'extend' => 'reload',
                    'className' => 'btn btn-dark',
                    'text' => "<i class='fa fa-sync-alt'></i>" . trans('dashboard/datatable.reload')
                ]*/
            ],

            'columnDefs' => [
                ['className' => 'text-center', 'targets' => '_all'],
            ],
        ];
    }

    /**
     * Define columns.
     */
    protected function getColumns(): array
    {
        return [
            Column::make('id')->title('#'),
            Column::make('code')->title('كود المريض'),
            Column::make('full_name')->title('اسم المريض')->searchable(false)->orderable(false),
            Column::make('phone')->title('الهاتف'),
            Column::make('gender')->title('النوع'),
            Column::make('birth_date')->title('تاريخ الميلاد'),
            Column::computed('actions')
                ->title('التحكم')
                ->exportable(false)
                ->printable(false)
                ->width(120)
                ->addClass('text-center'),
        ];
    }

    protected function filename(): string
    {
        return 'patients_' . date('YmdHis');
    }
}
