<?php

namespace App\Http\Controllers;

use App\Models\Appointment;
use App\Models\Invoices;
use App\Models\Patients;
use App\Models\User;
use App\Models\Visits;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class DashboardController extends Controller
{
    public function index()
    {
        $user = Auth::user();

        $clinicId = $user->clinic_id;
        $branchId = $user->branch_id;

        // إجمالي المرضى
        $totalPatients = Patients::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->count();

        // مرضى جدد اليوم
        $newPatientsToday = Patients::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->whereDate('created_at', today())
            ->count();

        // مواعيد اليوم
        $todayAppointments = Appointment::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->whereDate('appointment_date', today())
            ->count();

        // زيارات مكتملة اليوم
        $todayVisits = Visits::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->whereDate('visit_date', today())
            ->count();

        // دكاترة نشطين
        $activeDoctors = User::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->where('role', 'doctor')
            ->where('is_active', 1)
            ->count();

        // إجمالي فواتير هذا الشهر
        $monthInvoicesTotal = Invoices::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->whereMonth('date', now()->month)
            ->whereYear('date', now()->year)
            ->sum('total');

        // بيانات جراف عدد المواعيد آخر 7 أيام
        $appointmentsLast7Days = Appointment::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->whereDate('appointment_date', '>=', now()->subDays(6)->toDateString())
            ->selectRaw('DATE(appointment_date) as date, COUNT(*) as count')
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $appointmentsByStatus = Appointment::where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->selectRaw('status, COUNT(*) as count')
            ->groupBy('status')
            ->get();


        // آخر 10 مواعيد
        $latestAppointments = Appointment::with('patient', 'doctor.user')
            ->where('clinic_id', $clinicId)
            ->when($branchId, fn($q) => $q->where('branch_id', $branchId))
            ->orderByDesc('appointment_date')
            ->orderByDesc('start_time')
            ->take(10)
            ->get();

        return view('admin.index', [
            'user'                => $user,
            'totalPatients'       => $totalPatients,
            'newPatientsToday'    => $newPatientsToday,
            'todayAppointments'   => $todayAppointments,
            'todayVisits'         => $todayVisits,
            'activeDoctors'       => $activeDoctors,
            'monthInvoicesTotal'  => $monthInvoicesTotal,
            'appointmentsLast7Days' => $appointmentsLast7Days,
            'appointmentsByStatus'  => $appointmentsByStatus,
            'latestAppointments'  => $latestAppointments,
        ]);
    }
}
