<?php

namespace App\Http\Controllers\Invoices;

use App\Http\Controllers\Controller;
use App\Models\InvoiceItems;
use App\Models\Invoices;
use App\Models\Patients;
use App\Models\Services;
use App\Models\TreatmentPlans;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use App\Services\StockService;


class InvoiceController extends Controller
{
    /**
     * قائمة الفواتير
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        $query = Invoices::with([
            'patient:id,first_name,last_name,code,phone',
            'branch:id,name',
        ])
            ->where('clinic_id', $user->clinic_id)
            ->orderByDesc('date')
            ->orderByDesc('id');

        // فلترة بسيطة
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        if ($request->filled('patient_id')) {
            $query->where('patient_id', $request->patient_id);
        }

        if ($request->filled('from_date')) {
            $query->whereDate('date', '>=', $request->from_date);
        }

        if ($request->filled('to_date')) {
            $query->whereDate('date', '<=', $request->to_date);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('invoice_number', 'like', "%{$search}%")
                    ->orWhereHas('patient', function ($q2) use ($search) {
                        $q2->where('first_name', 'like', "%{$search}%")
                            ->orWhere('last_name', 'like', "%{$search}%")
                            ->orWhere('phone', 'like', "%{$search}%")
                            ->orWhere('code', 'like', "%{$search}%");
                    });
            });
        }

        $invoices = $query->paginate(20);

        return view('admin.invoices.index', compact('invoices'));
    }

    /**
     * صفحة إنشاء فاتورة جديدة
     * - ممكن تيجي من صفحة المريض أو خطة علاج
     *   /invoices/create?patient_id=XX&treatment_plan_id=YY
     */
    public function create(Request $request)
    {
        $user = Auth::user();

        $patientId       = $request->input('patient_id');
        $treatmentPlanId = $request->input('treatment_plan_id');

        $patient       = null;
        $treatmentPlan = null;
        $planItems     = collect();

        if ($patientId) {
            $patient = Patients::where('clinic_id', $user->clinic_id)
                ->findOrFail($patientId);
        }

        if ($treatmentPlanId) {
            $treatmentPlan = TreatmentPlans::with([
                'patient',
                'items.service',
            ])
                ->whereHas('patient', function ($q) use ($user) {
                    $q->where('clinic_id', $user->clinic_id);
                })
                ->findOrFail($treatmentPlanId);

            if (! $patient) {
                $patient = $treatmentPlan->patient;
            }

            $planItems = $treatmentPlan->items->map(function ($item) {
                return [
                    'service_id'   => $item->service_id,
                    'service_name' => optional($item->service)->name,
                    'description'  => $item->notes ?? '',
                    'quantity'     => (int) $item->quantity,
                    'price'        => (float) $item->price,
                    'discount'     => (float) $item->discount,
                ];
            })->values();
        }

        $services = Services::where('clinic_id', $user->clinic_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $servicesForJs = $services->map(function ($s) {
            return [
                'id'    => $s->id,
                'name'  => $s->name,
                'price' => (float) $s->default_price,
            ];
        })->values();

        // في حالة create مافيش فاتورة أو بنود سابقة
        $invoice          = null;
        $invoiceItemsForJs = [];

        return view('admin.invoices.create', compact(
            'patient',
            'treatmentPlan',
            'planItems',
            'services',
            'servicesForJs',
            'invoice',
            'invoiceItemsForJs'
        ));
    }

    /**
     * حفظ فاتورة جديدة
     */
    public function store(Request $request, StockService $stockService)
    {
        $user = Auth::user();

        // 1) Validation
        $data = $request->validate([
            'patient_id'        => ['required', 'integer', 'exists:patients,id'],
            'treatment_plan_id' => ['nullable', 'integer', 'exists:treatment_plans,id'],
            'visit_id'          => ['nullable', 'integer', 'exists:visits,id'],
            'date'              => ['required', 'date'],
            'payment_due_date'  => ['nullable', 'date'],
            'tax_percent'       => ['nullable', 'numeric', 'min:0'],
            'paid_amount'       => ['nullable', 'numeric', 'min:0'],
            'status'            => ['nullable', 'in:unpaid,partial,paid,refunded'],

            'items'                   => ['required', 'array', 'min:1'],
            'items.*.service_id'      => ['nullable', 'integer', 'exists:services,id'],
            'items.*.description'     => ['required', 'string', 'max:255'],
            'items.*.quantity'        => ['required', 'numeric', 'min:1'],
            'items.*.unit_price'      => ['required', 'numeric', 'min:0'],
            'items.*.discount'        => ['nullable', 'numeric', 'min:0'],
        ], [
            'patient_id.required' => 'من فضلك اختر المريض.',
            'items.required'      => 'يجب إضافة بند واحد على الأقل للفاتورة.',
        ]);

        // 2) تأكيد إن المريض تابع لنفس العيادة
        /** @var Patients $patient */
        $patient = Patients::where('clinic_id', $user->clinic_id)
            ->findOrFail($data['patient_id']);

        $treatmentPlanId = $data['treatment_plan_id'] ?? null;
        if ($treatmentPlanId) {
            // تأكيد أن خطة العلاج لنفس المريض ولنفس العيادة
            TreatmentPlans::where('id', $treatmentPlanId)
                ->where('patient_id', $patient->id)
                ->whereHas('patient', function ($q) use ($user) {
                    $q->where('clinic_id', $user->clinic_id);
                })
                ->firstOrFail();
        }

        // 3) تحضير البنود + حساب Subtotal و Discount
        $itemsInput = $data['items'];
        $itemsRows  = [];

        $subtotal      = 0;
        $totalDiscount = 0;

        foreach ($itemsInput as $row) {
            $qty       = (float) $row['quantity'];
            $price     = (float) $row['unit_price'];
            $discount  = isset($row['discount']) ? (float) $row['discount'] : 0.0;

            $lineBase = $qty * $price;
            if ($discount > $lineBase) {
                $discount = $lineBase;
            }

            $lineTotal = $lineBase - $discount;

            $subtotal      += $lineBase;
            $totalDiscount += $discount;

            $itemsRows[] = [
                'service_id'  => $row['service_id'] ?? null,
                'description' => $row['description'],
                'quantity'    => $qty,
                'unit_price'  => $price,
                'discount'    => $discount,
                'line_total'  => $lineTotal,
            ];
        }

        $taxPercent    = isset($data['tax_percent']) ? (float) $data['tax_percent'] : 0.0;
        $afterDiscount = max($subtotal - $totalDiscount, 0);
        $taxAmount     = $taxPercent > 0 ? round($afterDiscount * $taxPercent / 100, 2) : 0.0;
        $total         = $afterDiscount + $taxAmount;

        $paidAmount = isset($data['paid_amount']) ? (float) $data['paid_amount'] : 0.0;
        if ($paidAmount > $total) {
            $paidAmount = $total;
        }

        // 4) تحديد حالة الفاتورة لو المستخدم ما بعتهاش
        $status = $data['status'] ?? null;
        if (! $status) {
            if ($total == 0) {
                $status = 'paid';
            } elseif ($paidAmount == 0) {
                $status = 'unpaid';
            } elseif ($paidAmount < $total) {
                $status = 'partial';
            } else {
                $status = 'paid';
            }
        }

        // 5) تحميل الخدمات (لأجل استهلاك المخزون)
        $serviceIds = collect($itemsRows)
            ->pluck('service_id')
            ->filter()
            ->unique()
            ->values();

        $services = Services::with('inventoryLinks') // علاقة تربط الخدمة بعناصر المخزون + quantity_per_service
        ->whereIn('id', $serviceIds)
            ->get()
            ->keyBy('id');

        // 6) إنشاء الفاتورة + البنود + استهلاك المخزون داخل Transaction
        DB::transaction(function () use (
            $user,
            $patient,
            $data,
            $treatmentPlanId,
            $subtotal,
            $totalDiscount,
            $afterDiscount,
            $taxPercent,
            $taxAmount,
            $total,
            $paidAmount,
            $status,
            $itemsRows,
            $services,
            $stockService
        ) {
            $invoiceNumber = 'INV-' . $user->clinic_id . '-' . now()->format('YmdHis') . '-' . rand(100, 999);

            /** @var Invoices $invoice */
            $invoice = Invoices::create([
                'clinic_id'        => $user->clinic_id,
                'branch_id'        => $user->branch_id,
                'patient_id'       => $patient->id,
                'visit_id'         => $data['visit_id'] ?? null,
                'treatment_plan_id'=> $treatmentPlanId,
                'invoice_number'   => $invoiceNumber,
                'date'             => $data['date'],
                'payment_due_date' => $data['payment_due_date'] ?? null,

                'subtotal'     => $subtotal,
                'discount'     => $totalDiscount,
                'tax_percent'  => $taxPercent,
                'tax_amount'   => $taxAmount,
                'total'        => $total,
                'paid_amount'  => $paidAmount,
                'status'       => $status,
            ]);

            // البنود
            $createdItems = [];
            foreach ($itemsRows as $row) {
                $row['invoice_id'] = $invoice->id;
                $createdItems[]    = InvoiceItems::create($row);
            }

            // استهلاك المخزون لكل بند خدمة
            foreach ($createdItems as $invItem) {
                if (! $invItem->service_id) {
                    continue;
                }

                $service = $services->get($invItem->service_id);
                if (! $service || $service->inventoryLinks->isEmpty()) {
                    continue;
                }

                foreach ($service->inventoryLinks as $link) {
                    $qtyToConsume = $invItem->quantity * $link->quantity_per_service;
                    if ($qtyToConsume <= 0) {
                        continue;
                    }

                    $stockService->decreaseStock(
                        inventoryItemId: $link->inventory_item_id,
                        quantity:        $qtyToConsume,
                        branchId:        $user->branch_id,
                        reason:          'استهلاك خدمة من فاتورة',
                        referenceType:   'invoice',
                        referenceId:     $invoice->id,
                    );
                }
            }

            // (اختياري) لو حابب تسجل الدفعة الأولية كـ Payment، تقدر تضيف هنا
        });

        return redirect()
            ->route('invoices.index')
            ->with('success', 'تم إنشاء الفاتورة بنجاح.');
    }

    /**
     * عرض فاتورة واحدة
     */
    public function show($id)
    {
        $user = Auth::user();

        $invoice = Invoices::with([
            'patient',
            'items.service',
            'payments',
        ])
            ->where('clinic_id', $user->clinic_id)
            ->findOrFail($id);

        $remaining = max($invoice->total - $invoice->paid_amount, 0);

        return view('admin.invoices.show', compact('invoice', 'remaining'));
    }

    /**
     * صفحة تعديل فاتورة
     * (بنستخدم نفس Blade بتاع create مع وجود $invoice)
     */
    public function edit($id)
    {
        $user = Auth::user();

        $invoice = Invoices::with(['patient', 'treatmentPlan', 'items.service'])
            ->where('clinic_id', $user->clinic_id)
            ->findOrFail($id);

        $patient       = $invoice->patient;
        $treatmentPlan = $invoice->treatmentPlan;

        $services = Services::where('clinic_id', $user->clinic_id)
            ->where('is_active', true)
            ->orderBy('name')
            ->get();

        $servicesForJs = $services->map(function ($s) {
            return [
                'id'    => $s->id,
                'name'  => $s->name,
                'price' => (float) $s->default_price,
            ];
        })->values();

        $invoiceItemsForJs = $invoice->items->map(function ($item) {
            return [
                'service_id'   => $item->service_id,
                'service_name' => optional($item->service)->name,
                'description'  => $item->description,
                'quantity'     => (int) $item->quantity,
                'price'        => (float) $item->unit_price,
                'discount'     => (float) $item->discount,
            ];
        })->values();

        $planItems = collect(); // مش محتاجينها في edit

        return view('admin.invoices.create', compact(
            'patient',
            'treatmentPlan',
            'services',
            'servicesForJs',
            'planItems',
            'invoice',
            'invoiceItemsForJs'
        ));
    }

    /**
     * حفظ تعديل فاتورة
     * (بدون لعب في المخزون هنا – تقدر تزود rollback لو حابب)
     */
    public function update(Request $request, $id)
    {
        $user = Auth::user();

        $invoice = Invoices::where('clinic_id', $user->clinic_id)
            ->findOrFail($id);

        $data = $request->validate([
            'patient_id'        => ['required', 'integer', 'exists:patients,id'],
            'treatment_plan_id' => ['nullable', 'integer', 'exists:treatment_plans,id'],
            'visit_id'          => ['nullable', 'integer', 'exists:visits,id'],
            'date'              => ['required', 'date'],
            'payment_due_date'  => ['nullable', 'date'],
            'tax_percent'       => ['nullable', 'numeric', 'min:0'],
            'paid_amount'       => ['nullable', 'numeric', 'min:0'],
            'status'            => ['nullable', 'in:unpaid,partial,paid,refunded'],

            'items'                   => ['required', 'array', 'min:1'],
            'items.*.service_id'      => ['nullable', 'integer', 'exists:services,id'],
            'items.*.description'     => ['required', 'string', 'max:255'],
            'items.*.quantity'        => ['required', 'numeric', 'min:1'],
            'items.*.unit_price'      => ['required', 'numeric', 'min:0'],
            'items.*.discount'        => ['nullable', 'numeric', 'min:0'],
        ]);

        $patient = Patients::where('clinic_id', $user->clinic_id)
            ->findOrFail($data['patient_id']);

        $itemsInput = $data['items'];
        $itemsRows  = [];

        $subtotal      = 0;
        $totalDiscount = 0;

        foreach ($itemsInput as $row) {
            $qty       = (float) $row['quantity'];
            $price     = (float) $row['unit_price'];
            $discount  = isset($row['discount']) ? (float) $row['discount'] : 0.0;

            $lineBase = $qty * $price;
            if ($discount > $lineBase) {
                $discount = $lineBase;
            }

            $lineTotal = $lineBase - $discount;

            $subtotal      += $lineBase;
            $totalDiscount += $discount;

            $itemsRows[] = [
                'service_id'  => $row['service_id'] ?? null,
                'description' => $row['description'],
                'quantity'    => $qty,
                'unit_price'  => $price,
                'discount'    => $discount,
                'line_total'  => $lineTotal,
            ];
        }

        $taxPercent    = isset($data['tax_percent']) ? (float) $data['tax_percent'] : 0.0;
        $afterDiscount = max($subtotal - $totalDiscount, 0);
        $taxAmount     = $taxPercent > 0 ? round($afterDiscount * $taxPercent / 100, 2) : 0.0;
        $total         = $afterDiscount + $taxAmount;

        $paidAmount = isset($data['paid_amount']) ? (float) $data['paid_amount'] : 0.0;
        if ($paidAmount > $total) {
            $paidAmount = $total;
        }

        $status = $data['status'] ?? null;
        if (! $status) {
            if ($total == 0) {
                $status = 'paid';
            } elseif ($paidAmount == 0) {
                $status = 'unpaid';
            } elseif ($paidAmount < $total) {
                $status = 'partial';
            } else {
                $status = 'paid';
            }
        }

        DB::transaction(function () use (
            $invoice,
            $patient,
            $data,
            $subtotal,
            $totalDiscount,
            $taxPercent,
            $taxAmount,
            $total,
            $paidAmount,
            $status,
            $itemsRows
        ) {
            $invoice->update([
                'patient_id'       => $patient->id,
                'visit_id'         => $data['visit_id'] ?? null,
                'treatment_plan_id'=> $data['treatment_plan_id'] ?? null,
                'date'             => $data['date'],
                'payment_due_date' => $data['payment_due_date'] ?? null,

                'subtotal'     => $subtotal,
                'discount'     => $totalDiscount,
                'tax_percent'  => $taxPercent,
                'tax_amount'   => $taxAmount,
                'total'        => $total,
                'paid_amount'  => $paidAmount,
                'status'       => $status,
            ]);

            $invoice->items()->delete();

            foreach ($itemsRows as $row) {
                $row['invoice_id'] = $invoice->id;
                InvoiceItems::create($row);
            }

            /**
             * ملاحظة:
             * لو حابب تزبط المخزون عند التعديل (ترجع القديم + تستهلك الجديد)،
             * تقدر هنا:
             *  - تجيب stock_movements لـ reference_type = 'invoice' & reference_id = $invoice->id
             *  - ترجّع الكميات للمخزون
             *  - تحذف الحركات القديمة
             *  - تعيد استهلاك المخزون على البنود الجديدة (زي store)
             */
        });

        return redirect()
            ->route('invoices.show', $invoice->id)
            ->with('success', 'تم تحديث الفاتورة بنجاح.');
    }

    /**
     * حذف فاتورة (Soft Delete)
     */
    public function destroy($id)
    {
        $user = Auth::user();

        $invoice = Invoices::where('clinic_id', $user->clinic_id)
            ->withCount('payments')
            ->findOrFail($id);

        // لو في مدفوعات، ممكن تمنع الحذف أو تتعامل حسب الـ Policy عندك
        if ($invoice->payments_count > 0) {
            return redirect()
                ->back()
                ->with('error', 'لا يمكن حذف فاتورة عليها مدفوعات.');
        }

        $invoice->delete();

        return redirect()
            ->route('invoices.index')
            ->with('success', 'تم حذف الفاتورة بنجاح.');
    }

    /**
     * طباعة الفاتورة
     */
    public function print($id)
    {
        $user = Auth::user();

        $invoice = Invoices::with([
            'patient',
            'clinic',
            'branch',
            'items.service',
        ])
            ->where('clinic_id', $user->clinic_id)
            ->findOrFail($id);

        return view('admin.invoices.print', compact('invoice'));
    }

    public function ajaxPatientPlans(Request $request)
    {
        $user      = Auth::user();
        $patientId = $request->input('patient_id');

        $plans = TreatmentPlans::where('patient_id', $patientId)
            ->whereHas('patient', function ($q) use ($user) {
                $q->where('clinic_id', $user->clinic_id);
            })
            ->orderByDesc('created_at')
            ->get();

        $results = $plans->map(function ($plan) {
            return [
                'id'   => $plan->id,
                'text' => ($plan->title ?? ('خطة #' . $plan->id)) . ' — الحالة: ' . $plan->status,
            ];
        });

        return response()->json($results);
    }

    public function ajaxPlanItems(Request $request)
    {
        $user      = Auth::user();
        $planId    = $request->input('treatment_plan_id');

        $plan = TreatmentPlans::with(['items.service', 'patient'])
            ->where('id', $planId)
            ->whereHas('patient', function ($q) use ($user) {
                $q->where('clinic_id', $user->clinic_id);
            })
            ->firstOrFail();

        $items = $plan->items->map(function ($item) {
            return [
                'service_id'   => $item->service_id,
                'service_name' => optional($item->service)->name,
                'description'  => $item->notes ?? '',
                'quantity'     => (int) $item->quantity,
                'price'        => (float) $item->price,
                'discount'     => (float) $item->discount,
            ];
        })->values();

        return response()->json([
            'plan'  => [
                'id'     => $plan->id,
                'title'  => $plan->title,
                'status' => $plan->status,
            ],
            'items' => $items,
        ]);
    }
}
