<?php

namespace App\Http\Controllers\Patients;

use App\Http\Controllers\Controller;
use App\Models\Appointment;
use App\Models\Attachments;
use App\Models\Patients;
use App\Models\Prescriptions;
use App\Models\Visits;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class AttachmentController extends Controller
{
    public function store(Request $request)
    {
        $patientId = $request->input('patient_id', $request->input('id'));
        $patient   = Patients::findOrFail($patientId);

        $this->authorizePatient($patient);

        $formType = $request->input('form_type');

        if ($formType === 'attachment') {

            $data = $request->validate([
                'file'        => ['required', 'file', 'max:10240'],
                'file_type'   => ['nullable', 'string', 'max:255'],
                'description' => ['nullable', 'string', 'max:500'],
            ], [
                'file.required' => 'من فضلك اختر ملف للرفع.',
                'file.file'     => 'الملف غير صالح.',
                'file.max'      => 'حجم الملف يجب ألا يزيد عن 10 ميجابايت.',
            ]);

            $path = $request->file('file')->store('patients/' . $patient->id, 'public');
            $filePath = 'storage/' . $path;

            Attachments::create([
                'patient_id'  => $patient->id,
                'visit_id'    => null,
                'file_path'   => $filePath,
                'file_type'   => $data['file_type'] ?? null,
                'description' => $data['description'] ?? null,
            ]);

            return redirect()
                ->route('patients.show', $patient->id)
                ->with('success', 'تم رفع المرفق بنجاح.');

        }

        if ($formType === 'appointment') {

            $data = $request->validate([
                'appointment_date' => ['required', 'date'],
                'start_time'       => ['required'],
                'doctor_id'        => ['required', 'integer'], // تقدر تزود exists:doctor_profiles,id
            ], [
                'appointment_date.required' => 'من فضلك اختر تاريخ الموعد.',
                'start_time.required'       => 'من فضلك اختر وقت الموعد.',
                'doctor_id.required'        => 'من فضلك اختر الطبيب.',
            ]);

            $user = Auth::user();

            Appointment::create([
                'clinic_id'        => $user->clinic_id,
                'branch_id'        => $user->branch_id,
                'patient_id'       => $patient->id,
                'doctor_id'        => $data['doctor_id'],
                'appointment_date' => $data['appointment_date'],
                'start_time'       => $data['start_time'],
                'status'           => 'confirmed',
            ]);

            return redirect()
                ->route('patients.show', $patient->id)
                ->with('success', 'تم حجز الموعد بنجاح.');

        }

        if ($formType === 'prescription') {

            $data = $request->validate([

                'visit_id'  => ['required', 'exists:visits,id'],
                'notes'     => ['required', 'string'],
            ], [
                'doctor_id.required' => 'من فضلك اختر الطبيب.',
            ]);


            $Visits = Visits::findorfail($data['visit_id']);

            Prescriptions::create([
                'patient_id' => $patient->id,
                'doctor_id'  => $Visits->doctor_id,
                'visit_id'   => $data['visit_id'] ?? null,
                'notes'      => $data['notes'] ?? null,
            ]);

            return redirect()
                ->route('patients.show', $patient->id)
                ->with('success', 'تم حفظ الروشتة بنجاح.');
        }

        if ($formType === 'visit') {
            $data = $request->validate([
                'visit_date'       => ['required', 'date'],
                'doctor_id'        => ['required', 'integer'],
                'chief_complaint'  => ['nullable', 'string'],
                'notes'            => ['nullable', 'string'],
            ], [
                'visit_date.required' => 'من فضلك اختر تاريخ الزيارة.',
                'doctor_id.required'  => 'من فضلك اختر الطبيب.',
            ]);

            $user = Auth::user();

            Visits::create([
                'clinic_id'       => $user->clinic_id,
                'branch_id'       => $user->branch_id,
                'patient_id'      => $patient->id,
                'doctor_id'       => $data['doctor_id'],
                'visit_date'      => $data['visit_date'],
                'chief_complaint' => $data['chief_complaint'] ?? null,
                'notes'           => $data['notes'] ?? null,
                'status'          => 'open',
            ]);

            return redirect()
                ->route('patients.show', $patient->id)
                ->with('success', 'تم إضافة زيارة جديدة للمريض.');
        }

        return redirect()
            ->route('patients.show', $patient->id)
            ->with('error', 'نوع الطلب غير معروف.');
    }

    /**
     * Remove the specified attachment from storage.
     */
    public function destroy(Attachments $attachment)
    {
        $patient = $attachment->patient;

        $this->authorizePatient($patient);

        if ($attachment->file_path) {
            $publicPath = str_replace('storage/', '', $attachment->file_path); // يتحول لـ patients/...
            Storage::disk('public')->delete($publicPath);
        }

        $attachment->delete();

        return redirect()
            ->route('patients.show', $patient->id)
            ->with('success', 'تم حذف المرفق بنجاح.');
    }





    protected function authorizePatient(Patients $patient): void
    {
        $user = Auth::user();

        if (!$user || !$user->clinic_id) {
            abort(403, 'غير مصرح لك بالوصول إلى هذا المريض.');
        }

        if ($patient->clinic_id !== $user->clinic_id) {
            abort(404);
        }

        if ($user->role !== 'owner' && $user->branch_id && $patient->branch_id !== $user->branch_id) {
            abort(404);
        }
    }
}
