@extends('admin.layouts.master')

@section('title', isset($invoice) ? 'تعديل فاتورة' : 'إضافة فاتورة جديدة')

@section('css')
    <link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet"/>
@endsection

@section('content')

    <div class="content-header d-flex justify-content-between align-items-center mb-3">
        <h5 class="mb-0">
            {{ isset($invoice) ? 'تعديل فاتورة #' . $invoice->invoice_number : 'إضافة فاتورة جديدة' }}
        </h5>

        @if(isset($patient))
            <a href="{{ route('patients.show', $patient->id) }}" class="btn btn-sm btn-outline-secondary">
                الرجوع لملف المريض
            </a>
        @endif
    </div>

    @if($errors->any())
        <div class="alert alert-danger">
            <strong>من فضلك صحح الأخطاء التالية:</strong>
            <ul class="mb-0">
                @foreach($errors->all() as $err)
                    <li>{{ $err }}</li>
                @endforeach
            </ul>
        </div>
    @endif

    <form action="{{ isset($invoice)
        ? route('invoices.update', $invoice->id)
        : route('invoices.store') }}"
          method="POST">
        @csrf
        @if(isset($invoice))
            @method('PUT')
        @endif

        <div class="row gy-4">

            {{-- بيانات أساسية --}}
            <div class="col-12">
                <div class="card">
                    <div class="card-header">
                        <h6 class="mb-0">بيانات الفاتورة الأساسية</h6>
                    </div>
                    <div class="card-body">
                        <div class="row g-3">

                            {{-- المريض --}}
                            <div class="col-md-4">
                                <label class="form-label">المريض *</label>

                                <select id="patient-select" name="patient_id" class="form-control" required>
                                    @if(isset($patient))
                                        <option value="{{ $patient->id }}" selected>
                                            {{ $patient->first_name }} {{ $patient->last_name }}
                                            — {{ $patient->phone }}
                                        </option>
                                    @elseif(old('patient_id'))
                                        <option value="{{ old('patient_id') }}" selected>
                                            مريض رقم #{{ old('patient_id') }}
                                        </option>
                                    @endif
                                </select>

                                <small id="patient-info" class="text-muted d-block mt-1">
                                    @if(isset($patient))
                                        <span class="text-success fw-bold">
                                            {{ $patient->first_name }} {{ $patient->last_name }}
                                        </span><br>
                                        الهاتف: {{ $patient->phone }}<br>
                                        @if($patient->code)
                                            كود المريض: {{ $patient->code }}
                                        @endif
                                    @endif
                                </small>
                            </div>

                            {{-- خطة علاج من نفس المريض --}}
                            <div class="col-md-4">
                                <label class="form-label">خطة العلاج (اختياري)</label>

                                <select id="treatment-plan-select" name="treatment_plan_id"
                                        class="form-select"
                                    {{ isset($treatmentPlan) ? '' : 'disabled' }}>
                                    @if(isset($treatmentPlan))
                                        <option value="{{ $treatmentPlan->id }}" selected>
                                            {{ $treatmentPlan->title ?? ('خطة #' . $treatmentPlan->id) }}
                                            — الحالة: {{ $treatmentPlan->status }}
                                        </option>
                                    @else
                                        <option value="">اختر خطة علاج بعد اختيار المريض</option>
                                    @endif
                                </select>
                            </div>

                            {{-- رقم الزيارة (اختياري) --}}
                            <div class="col-md-4">
                                <label class="form-label">رقم الزيارة (اختياري)</label>
                                <input type="number" name="visit_id" class="form-control"
                                       value="{{ old('visit_id', isset($invoice) ? $invoice->visit_id : '') }}">
                            </div>

                            {{-- تاريخ الفاتورة --}}
                            <div class="col-md-4">
                                <label class="form-label">تاريخ الفاتورة *</label>
                                <input type="date" name="date" class="form-control"
                                       value="{{ old('date', isset($invoice) ? $invoice->date->format('Y-m-d') : now()->toDateString()) }}"
                                       required>
                            </div>

                            {{-- تاريخ الاستحقاق --}}
                            <div class="col-md-4">
                                <label class="form-label">تاريخ الاستحقاق (اختياري)</label>
                                <input type="date" name="payment_due_date" class="form-control"
                                       value="{{ old('payment_due_date', isset($invoice) && $invoice->payment_due_date
                                                ? $invoice->payment_due_date->format('Y-m-d')
                                                : '') }}">
                            </div>

                            {{-- الضريبة --}}
                            <div class="col-md-2">
                                <label class="form-label">الضريبة %</label>
                                <input type="number" step="0.01" min="0" name="tax_percent"
                                       class="form-control"
                                       id="tax-percent-input"
                                       value="{{ old('tax_percent', isset($invoice) ? $invoice->tax_percent : 0) }}">
                            </div>

                            {{-- المدفوع حالياً --}}
                            <div class="col-md-2">
                                <label class="form-label">المدفوع الآن</label>
                                <input type="number" step="0.01" min="0" name="paid_amount"
                                       class="form-control"
                                       value="{{ old('paid_amount', isset($invoice) ? $invoice->paid_amount : 0) }}"
                                       id="paid-amount-input">
                            </div>

                            {{-- حالة الفاتورة --}}
                            <div class="col-md-2">
                                <label class="form-label">حالة الفاتورة</label>
                                @php
                                    $currentStatus = old('status', isset($invoice) ? $invoice->status : 'unpaid');
                                @endphp
                                <select name="status" class="form-select">
                                    <option value="unpaid"   {{ $currentStatus == 'unpaid'   ? 'selected' : '' }}>غير مدفوعة</option>
                                    <option value="partial"  {{ $currentStatus == 'partial'  ? 'selected' : '' }}>مدفوعة جزئياً</option>
                                    <option value="paid"     {{ $currentStatus == 'paid'     ? 'selected' : '' }}>مدفوعة بالكامل</option>
                                    <option value="refunded" {{ $currentStatus == 'refunded' ? 'selected' : '' }}>مسترجعة</option>
                                </select>
                            </div>

                        </div>
                    </div>
                </div>
            </div>

            {{-- بنود الفاتورة (من خطة العلاج) --}}
            <div class="col-12">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h6 class="mb-0">بنود الفاتورة (تأتي من خطة العلاج)</h6>
                        <button type="button" class="btn btn-sm btn-success" id="btn-add-row">
                            إضافة بند يدوي (اختياري)
                        </button>
                    </div>

                    <div class="card-body">

                        <div class="table-responsive">
                            <table class="table table-sm align-middle" id="items-table">
                                <thead>
                                <tr>
                                    <th style="width: 5%;">#</th>
                                    <th style="width: 25%;">الخدمة</th>
                                    <th>الوصف</th>
                                    <th style="width: 8%;">الكمية</th>
                                    <th style="width: 13%;">سعر الوحدة</th>
                                    <th style="width: 13%;">الخصم</th>
                                    <th style="width: 13%;">الإجمالي</th>
                                    <th style="width: 8%;">حذف</th>
                                </tr>
                                </thead>
                                <tbody id="items-rows">
                                {{-- يتم تعبئتها بالـ JS من خطة العلاج أو من invoiceItems --}}
                                </tbody>
                            </table>
                        </div>

                        {{-- ملخص الأرقام Front فقط --}}
                        <div class="row mt-3">
                            <div class="col-md-6 small text-muted">
                                * سيتم حساب القيم النهائية على السيرفر أيضًا لضمان الدقة.
                                هذا الملخص للعرض فقط.
                            </div>

                            <div class="col-md-6">
                                <table class="table table-sm mb-0">
                                    <tr>
                                        <td class="text-start">الإجمالي قبل الخصم:</td>
                                        <td class="text-end">
                                            <span id="summary-subtotal">0.00</span>
                                            <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="14"
                                                 height="14" alt="ر.س">
                                        </td>
                                    </tr>
                                    <tr>
                                        <td class="text-start">إجمالي الخصم:</td>
                                        <td class="text-end">
                                            <span id="summary-discount">0.00</span>
                                            <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="14"
                                                 height="14" alt="ر.س">
                                        </td>
                                    </tr>
                                    <tr>
                                        <td class="text-start">الإجمالي بعد الخصم:</td>
                                        <td class="text-end">
                                            <span id="summary-after-discount">0.00</span>
                                            <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="14"
                                                 height="14" alt="ر.س">
                                        </td>
                                    </tr>
                                    <tr>
                                        <td class="text-start">
                                            الضريبة (<span id="summary-tax-percent">0</span>%):
                                        </td>
                                        <td class="text-end">
                                            <span id="summary-tax-amount">0.00</span>
                                            <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="14"
                                                 height="14" alt="ر.س">
                                        </td>
                                    </tr>
                                    <tr>
                                        <td class="text-start fw-bold">الإجمالي النهائي:</td>
                                        <td class="text-end fw-bold">
                                            <span id="summary-total">0.00</span>
                                            <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="14"
                                                 height="14" alt="ر.س">
                                        </td>
                                    </tr>
                                    <tr>
                                        <td class="text-start">المدفوع الآن:</td>
                                        <td class="text-end">
                                            <span id="summary-paid">0.00</span>
                                            <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="14"
                                                 height="14" alt="ر.س">
                                        </td>
                                    </tr>
                                    <tr>
                                        <td class="text-start">المتبقي:</td>
                                        <td class="text-end">
                                            <span id="summary-due">0.00</span>
                                            <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="14"
                                                 height="14" alt="ر.س">
                                        </td>
                                    </tr>
                                </table>
                            </div>
                        </div>

                    </div>

                    <div class="card-footer text-end">
                        <button type="submit" class="btn btn-primary">
                            {{ isset($invoice) ? 'تحديث الفاتورة' : 'حفظ الفاتورة' }}
                        </button>
                    </div>
                </div>
            </div>

        </div>
    </form>

@endsection

@section('js')
    {{-- Select2 --}}
    <script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>

    <script>
        $(document).ready(function () {

            // اختيار المريض
            $('#patient-select').select2({
                placeholder: "ابحث باسم المريض / الهاتف / الكود",
                minimumInputLength: 1,
                dir: "rtl",
                width: "100%",
                ajax: {
                    url: "{{ route('patients.ajax.search') }}",
                    dataType: 'json',
                    delay: 250,
                    data: function (params) {
                        return { q: params.term };
                    },
                    processResults: function (data) {
                        return { results: data };
                    }
                }
            });

            // عند اختيار المريض → عرض معلوماته + تحميل خطط العلاج
            $('#patient-select').on('select2:select', function (e) {
                let p = e.params.data;

                $('#patient-info').html(`
                    <span class="text-success fw-bold">${p.name}</span><br>
                    الهاتف: ${p.phone}<br>
                    كود المريض: ${p.code}
                `);

                loadTreatmentPlans(p.id);
            });

            // لو الصفحة فتحت وفيها مريض جاهز، نقدر تلقائيًا نحمّل الخطط لو حابب
            @if(isset($patient) && !isset($treatmentPlan))
            loadTreatmentPlans({{ $patient->id }});
            @endif

            // عند تغيير خطة العلاج من الـ select
            $('#treatment-plan-select').on('change', function () {
                const planId = $(this).val();
                if (!planId) {
                    clearItemsTable();
                    return;
                }
                loadPlanItems(planId);
            });

            function loadTreatmentPlans(patientId) {
                const $planSelect = $('#treatment-plan-select');
                $planSelect.prop('disabled', true).empty()
                    .append('<option value="">جاري تحميل الخطط...</option>');

                $.ajax({
                    url: "{{ route('invoices.ajax.plans') }}",
                    data: { patient_id: patientId },
                    success: function (res) {
                        $planSelect.empty().append('<option value="">اختر خطة علاج</option>');
                        res.forEach(function (plan) {
                            $planSelect.append(new Option(plan.text, plan.id, false, false));
                        });
                        $planSelect.prop('disabled', false);
                    },
                    error: function () {
                        $planSelect.empty().append('<option value="">حدث خطأ في تحميل الخطط</option>');
                    }
                });
            }

            function loadPlanItems(planId) {
                $.ajax({
                    url: "{{ route('invoices.ajax.plan-items') }}",
                    data: { treatment_plan_id: planId },
                    success: function (res) {
                        clearItemsTable();
                        if (Array.isArray(res.items)) {
                            res.items.forEach(function (item) {
                                addRowFromPlan(item);
                            });
                        }
                    },
                    error: function () {
                        alert('حدث خطأ أثناء تحميل بنود خطة العلاج.');
                    }
                });
            }

            function clearItemsTable() {
                const tbody = document.getElementById('items-rows');
                tbody.innerHTML = '';
                window.invoiceRowIndex = 0;
                recalc();
            }

            // نخلي الـ index global علشان نستخدمه جوه Ajax
            window.invoiceRowIndex = 0;

            const services      = @json($servicesForJs ?? []);
            const planItems     = @json($planItems ?? []);
            const invoiceItems  = @json($invoiceItemsForJs ?? []);

            function formatNumber(num) {
                return (parseFloat(num) || 0).toFixed(2);
            }

            // إضافة صف من خطة علاج (سلوك شبيه بـ addRow العادي)
            function addRowFromPlan(prefilled) {
                addRow(prefilled);
            }

            function addRow(prefilled = {}) {
                const tbody = document.getElementById('items-rows');
                const idx   = window.invoiceRowIndex;
                const tr    = document.createElement('tr');
                tr.dataset.index = idx;

                const serviceOptions = ['<option value="">بدون</option>'].concat(
                    services.map(s => {
                        const selected = prefilled.service_id && prefilled.service_id == s.id ? 'selected' : '';
                        return `
                            <option value="${s.id}" data-price="${s.price}" ${selected}>
                                ${s.name} - ${formatNumber(s.price)} ر.س
                            </option>
                        `;
                    })
                ).join('');

                tr.innerHTML = `
                    <td class="text-center align-middle">
                        <span class="row-number">${idx + 1}</span>
                    </td>

                    <td>
                        <select name="items[${idx}][service_id]"
                                class="form-select form-select-sm service-select">
                            ${serviceOptions}
                        </select>
                    </td>

                    <td>
                        <input type="text" name="items[${idx}][description]"
                               class="form-control form-control-sm desc-input"
                               placeholder="وصف الخدمة / الإجراء"
                               value="${prefilled.description || (prefilled.service_name || '')}">
                    </td>

                    <td>
                        <input type="number" min="1" step="1"
                               name="items[${idx}][quantity]"
                               class="form-control form-control-sm qty-input"
                               value="${prefilled.quantity || 1}">
                    </td>

                    <td>
                        <input type="number" min="0" step="0.01"
                               name="items[${idx}][unit_price]"
                               class="form-control form-control-sm price-input"
                               value="${(prefilled.price !== undefined) ? prefilled.price : 0}">
                    </td>

                    <td>
                        <input type="number" min="0" step="0.01"
                               name="items[${idx}][discount]"
                               class="form-control form-control-sm discount-input"
                               value="${prefilled.discount || 0}">
                    </td>

                    <td class="text-end align-middle">
                        <span class="line-total">0.00</span>
                        <img src="{{ asset('dash/Saudi_Riyal_Symbol.svg.png') }}" width="12" height="12" alt="ر.س">
                    </td>

                    <td class="text-center align-middle">
                        <button type="button" class="btn btn-sm btn-outline-danger btn-remove-row">
                            ×
                        </button>
                    </td>
                `;

                tbody.appendChild(tr);
                window.invoiceRowIndex++;
                attachRowEvents(tr);
                recalc();
            }

            function attachRowEvents(tr) {
                const serviceSelect = tr.querySelector('.service-select');
                const priceInput    = tr.querySelector('.price-input');
                const qtyInput      = tr.querySelector('.qty-input');
                const discountInput = tr.querySelector('.discount-input');
                const removeBtn     = tr.querySelector('.btn-remove-row');

                serviceSelect.addEventListener('change', function () {
                    const selected = this.options[this.selectedIndex];
                    const price = selected.getAttribute('data-price');
                    if (price !== null) {
                        priceInput.value = formatNumber(price);
                    }
                    recalc();
                });

                [priceInput, qtyInput, discountInput].forEach(input => {
                    input.addEventListener('input', recalc);
                });

                removeBtn.addEventListener('click', function () {
                    tr.remove();
                    updateRowNumbers();
                    recalc();
                });
            }

            function updateRowNumbers() {
                document.querySelectorAll('#items-rows tr').forEach((tr, idx) => {
                    const span = tr.querySelector('.row-number');
                    if (span) span.textContent = idx + 1;
                });
            }

            function recalc() {
                let subtotal      = 0;
                let totalDiscount = 0;

                document.querySelectorAll('#items-rows tr').forEach(tr => {
                    const qtyInput      = tr.querySelector('.qty-input');
                    const priceInput    = tr.querySelector('.price-input');
                    const discountInput = tr.querySelector('.discount-input');
                    const lineTotalSpan = tr.querySelector('.line-total');

                    const qty      = parseFloat(qtyInput.value) || 0;
                    const price    = parseFloat(priceInput.value) || 0;
                    let discount   = parseFloat(discountInput.value) || 0;

                    const lineBase = qty * price;
                    if (discount > lineBase) {
                        discount = lineBase;
                        discountInput.value = formatNumber(discount);
                    }

                    const lineTotal = lineBase - discount;

                    subtotal      += lineBase;
                    totalDiscount += discount;

                    lineTotalSpan.textContent = formatNumber(lineTotal);
                });

                const taxPercent = parseFloat($('#tax-percent-input').val()) || 0;
                const paidAmount = parseFloat($('#paid-amount-input').val()) || 0;

                let afterDiscount = subtotal - totalDiscount;
                if (afterDiscount < 0) afterDiscount = 0;

                let taxAmount = 0;
                if (taxPercent > 0) {
                    taxAmount = afterDiscount * (taxPercent / 100);
                }

                const total = afterDiscount + taxAmount;
                let due     = total - paidAmount;
                if (due < 0) due = 0;

                $('#summary-subtotal').text(formatNumber(subtotal));
                $('#summary-discount').text(formatNumber(totalDiscount));
                $('#summary-after-discount').text(formatNumber(afterDiscount));
                $('#summary-tax-percent').text(formatNumber(taxPercent));
                $('#summary-tax-amount').text(formatNumber(taxAmount));
                $('#summary-total').text(formatNumber(total));
                $('#summary-paid').text(formatNumber(paidAmount));
                $('#summary-due').text(formatNumber(due));
            }

            // زر إضافة بند يدوي لو حبيت
            $('#btn-add-row').on('click', function () {
                addRow();
            });

            $('#tax-percent-input').on('input', recalc);
            $('#paid-amount-input').on('input', recalc);

            // لو بنعدّل فاتورة → حمّل البنود من الفاتورة
            if (Array.isArray(invoiceItems) && invoiceItems.length > 0) {
                invoiceItems.forEach(function (item) {
                    addRow(item);
                });
            }
            // لو فاتورة جديدة من خطة علاج (coming from ?treatment_plan_id=)
            else if (Array.isArray(planItems) && planItems.length > 0) {
                planItems.forEach(function (item) {
                    addRow(item);
                });
            }
            // غير كده → مفيش ولا حاجة، استنى اختيار المريض وخطة العلاج
        });
    </script>
@endsection
